/*
 * MIDIUSB_test.ino
 *
 * Created: 4/6/2015 10:47:08 AM
 * Author: gurbrinder grewal
 * Modified by Arduino LLC (2015)
 */ 

// First parameter is the event type (0x09 = note on, 0x08 = note off).
// Second parameter is note-on/note-off, combined with the channel.
// Channel can be anything between 0-15. Typically reported to the user as 1-16.
// Third parameter is the note number (48 = middle C).
// Fourth parameter is the velocity (64 = normal, 127 = fastest).

#include <MIDIUSB.h>
#include <Adafruit_FreeTouch.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

// --- Config ---
#define N_TOUCH 6
#define THRESHOLD 500
#define SCREEN_WIDTH 128 // OLED display width, in pixels
#define SCREEN_HEIGHT 64 // OLED display height, in pixels
#define SCREEN_ADDRESS 0x3C // 0x3D or 0x3C depending on brand

// Q touch peripherals
uint8_t touch_pins[N_TOUCH] = {2, 3, 4, 5, 6, 7};

// MIDI mapping: one note per touch pad
// (C4, D4, E4, F4, G4, A4)
uint8_t note_map[N_TOUCH] = {60, 62, 64, 65, 67, 69};

// MIDI channel: 0–15 == DAW channels 1–16
const uint8_t MIDI_CH   = 0;
const uint8_t VELOCITY  = 100;

// --- Touch objects & state ---
Adafruit_FreeTouch* touch_devices[N_TOUCH];

// touch state
int touch_values[N_TOUCH] = {0, 0, 0, 0, 0, 0};
bool pin_touched_now[N_TOUCH] = {false, false, false, false, false, false};
bool pin_touched_past[N_TOUCH] = {false, false, false, false, false, false};

// --- MIDI helpers ---
static inline void sendNoteOn(uint8_t ch, uint8_t pitch, uint8_t vel) {
  midiEventPacket_t ev = {0x09, (uint8_t)(0x90 | ch), pitch, vel};
  MidiUSB.sendMIDI(ev);
}

static inline void sendNoteOff(uint8_t ch, uint8_t pitch, uint8_t vel) {
  midiEventPacket_t ev = {0x08, (uint8_t)(0x80 | ch), pitch, vel};
  MidiUSB.sendMIDI(ev);
}

static inline void sendControlChange(uint8_t ch, uint8_t cc, uint8_t value) {
  midiEventPacket_t ev = {0x0B, (uint8_t)(0xB0 | ch), cc, value};
  MidiUSB.sendMIDI(ev);
}

// --- NEW: Note name helper ---
const char* noteName(uint8_t midiNote) {
  static const char* names[] = {"C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"};
  int octave = (midiNote / 12) - 1;   // MIDI spec: C4 = 60
  int note   = midiNote % 12;
  static char buf[5];
  sprintf(buf, "%s%d", names[note], octave);
  return buf;
}

// --- Touch reading ---
void update_touch() {
  Ptc *ptc = ((Ptc *)PTC);

  for (int i = 0; i < N_TOUCH; i++) {
    // start
    touch_devices[i]->begin();

    // read
    touch_values[i] = touch_devices[i]->measure();

    // reset
    ptc->CTRLA.bit.ENABLE = 0;
    ptc->CTRLA.bit.SWRESET = 1;

    // update the state
    pin_touched_past[i] = pin_touched_now[i];
    pin_touched_now[i] = touch_values[i] > THRESHOLD;
  }
}

void print_touch() {
  char print_buffer[30];
  for (int i=0; i < N_TOUCH; i++) {
    sprintf(print_buffer, "%4d ", touch_values[i]);
    Serial.print(print_buffer);
  }
  Serial.println("");
}

// Display setup
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, -1, 1700000UL, 1700000UL);

void setup() {
  Serial.begin(0);
  
  // give the screen some time to power up
  delay(50);

  // initialize display
  display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS);
  display.clearDisplay();
  display.display();

  // text settings
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);

  // initialize capacitive touch peripherals
  for (int i = 0; i < N_TOUCH; i++) {
    touch_devices[i] = new Adafruit_FreeTouch(
      touch_pins[i], 
      OVERSAMPLE_1, 
      RESISTOR_100K, 
      FREQ_MODE_NONE
    );
  }
}  

void loop() {
  // 1) Update touch states
  update_touch();

  // 2) Send MIDI on edges for ALL pads (polyphonic)
  for (int i = 0; i < N_TOUCH; i++) {
    
    // Just pressed
    if (pin_touched_now[i] && !pin_touched_past[i]) {
      sendNoteOn(MIDI_CH, note_map[i], VELOCITY);
    }

    // Just released
    if (!pin_touched_now[i] && pin_touched_past[i]) {
      sendNoteOff(MIDI_CH, note_map[i], 0);
    }
  }

  // 3) Flush MIDI so the host actually receives it
  MidiUSB.flush();

  // 4) Optional: print raw values for tuning
  print_touch();

  // 5) Throttle loop (adjust as needed)
  delay(5);

  // --- DISPLAY CODE ---

  // clear the buffer
  display.clearDisplay();
  display.setCursor(0, 0);

  // Collect and show all pressed notes
  display.print("Notes: ");

  bool anyPressed = false;
  for (int i = 0; i < N_TOUCH; i++) {
    if (pin_touched_now[i]) {
      display.print(noteName(note_map[i]));  // prints MIDI note 
      display.print(" ");
      anyPressed = true;
    }
  }

  if (!anyPressed) {
    display.print("(none)");
  }

  // send buffer
  display.display();

  // (Optional) Read incoming MIDI if you care about it
  // midiEventPacket_t rx;
  // do {
  //   rx = MidiUSB.read();
  //   if (rx.header) {
  //     // Handle incoming if desired
  //   }
  // } while (rx.header);
}
